/*global define, require */
/*jslint white: true */

/*
	curve utils:

	This file implements the 2d time graph object.
*/

define ([	"src/math/curveUtils",	"src/math/mathUtils",	"src/utils"],
function(	curveUtils,				mathUtils,				utils) {
	'use strict';

	return function TimeGraph(inCurves) {
		var that = this;

		that.graphs = [];
		that.virtualTGraph = undefined;	// optionally defined.  Remaps t (in:0-1, out:0-1), used to control velocity.
		that.xMin = undefined;
		that.xMax = undefined;
		that.yMin = undefined;
		that.yMax = undefined;

		function throwBadTimeGraphError() {
			throw new Error("Incorrect time graph formulation.");
		}

		that.getXMin = function () {
			return this.xMin;
		};

		that.getXMax = function () {
			return this.xMax;
		};

		that.getYMin = function () {
			return this.yMin;
		};

		that.getYMax = function () {
			return this.yMax;
		};

		that.appendCurve = function (inCurve) {
			var	i = 0, xMin;

			xMin = inCurve[0][0];
			if (this.graphs.length === 0) {
				this.xMin = xMin;
				this.xMax = xMin;
				this.yMin = inCurve[0][1];
				this.yMax = inCurve[0][1];
			} else if (Math.abs(this.xMax - xMin) > 1e-6) {
				throwBadTimeGraphError();
			}
			for (i = 0; i < 4; i += 1) {
				if ((inCurve[i][0] < this.xMin) || (inCurve[i][0] < this.xMax)) {
					throwBadTimeGraphError();
				}
				this.xMax = inCurve[i][0];
				this.yMin = Math.min(this.yMin, inCurve[i][1]);
				this.yMax = Math.max(this.yMax, inCurve[i][1]);
			}
			this.graphs.push(utils.clone(true, [], inCurve));
		};

		that.appendLine = function (inLine) {
			this.appendCurve([inLine[0], inLine[0], inLine[1], inLine[1]]);
		};

		that.appendCurves = function (inCurves) {
			var	i = 0;

			for (i = 0; i < inCurves.length; i += 1) {
				this.appendCurve(inCurves[i]);
			}
		};

		function findEvalCurve(inTimeGraph, inX, inPreferLast) {
			var	foundCurve, curve, minIter = 0, maxIter = inTimeGraph.graphs.length, midIter, minComp, maxComp;

			inPreferLast = inPreferLast || true;
			// Perform a binary search to find the curve that intersects this x value.
			if (maxIter === 0) {
				throwBadTimeGraphError();
			} else if (maxIter === 1) {
				foundCurve = inTimeGraph.graphs[minIter];
			} else {
				do {
					midIter = Math.floor((maxIter + minIter) / 2);
					curve = inTimeGraph.graphs[midIter];

					minComp = curve[0][0] <= inX;
					maxComp = curve[3][0] >= inX;
					if (minComp && maxComp) {
						foundCurve = curve;
						if (inPreferLast) {
							// We found a curve, but for hold values, we'd prefer the last value.
							minIter = midIter + 1;
						} else {
							// We found a curve, but for hold values, we'd prefer the last value.
							minIter = midIter + 1;
						}
					} else if (!minComp && !maxComp) {
						throwBadTimeGraphError();
					} else if (!minComp) {
						maxIter = midIter;
					} else {
						minIter = midIter + 1;
					}
				} while (minIter < maxIter);
			}

			return foundCurve;
		}

		// optionally defined.  Maps 0-1 to 0-1, used to control velocity.
		that.setVirtualTGraph = function (inTimeGraph) {
			if (inTimeGraph !== undefined) {
				this.virtualTGraph = inTimeGraph.clone();
			} else {
				this.virtualTGraph = undefined;
			}
		};

		that.getVirtualTGraph = function () {
			return this.virtualTGraph;
		};

		that.getT = function (inT, inLoopType) {
			if (this.virtualTGraph) {
				inT = this.virtualTGraph.evaluateXFromT(inT, inLoopType);
				inT = this.virtualTGraph.evaluateYFromX(inT);
			} else {
				inT = mathUtils.loop(inT, inLoopType);
			}
			return inT;
		};

		// Evaluate based on X to produce a Y value.
		that.evaluateYFromX = function (inX) {
			var	evalY = 0, curve;
			if (inX < this.xMin) {
				inX = this.xMin;
			} else if (inX > this.xMax) {
				inX = this.xMax;
			}

			if (this.graphs.length >= 1) {
				curve = findEvalCurve(this, inX);
				evalY = curveUtils.evaluateYfromX(curve, inX);
			}

			return evalY;
		};

		// Evaluate based on T (0-1) to produce a X value.
		that.evaluateXFromT = function (inT, inLoopType) {
			var	t = this.getT(inT, inLoopType);
			return mathUtils.lerp(this.xMin, this.xMax, t);
		};

		that.evaluate = that.evaluateYFromX;

		that.sustainValue = function (inTime) {
			var	distSegment, value, lastTime, lastGraphSegment;
			lastGraphSegment = this.graphs[this.graphs.length - 1];
			if (lastGraphSegment) {
				lastTime = this.xMax;
				value = this.evaluate(lastTime);
				if (mathUtils.equalWithinEpsilon(lastGraphSegment[0][1], value) && mathUtils.equalWithinEpsilon(lastGraphSegment[1][1], value) &&
						mathUtils.equalWithinEpsilon(lastGraphSegment[2][1], value) && mathUtils.equalWithinEpsilon(lastGraphSegment[3][1], value)) {
					lastGraphSegment[3][0] = inTime;
					this.xMax = inTime;
				} else {
					lastGraphSegment = [[lastTime, value], [inTime, value]];
					this.appendLine(lastGraphSegment);
				}
			}
		};

		that.clone = function () {
			var newTimeGraph = utils.clone(true, {}, this);
			return newTimeGraph;
		};

		if (inCurves !== undefined) {
			that.appendCurves(inCurves);
		}

		return that;
	};
});
